<?php
/*******************************************************************************
 *  Copyright 2010 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at: http://aws.amazon.com/apache2.0
 *  This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 *  CONDITIONS OF ANY KIND, either express or implied. See the License for the
 *  specific language governing permissions and limitations under the License.
 * *****************************************************************************
 */


/**
 *  @see CheckoutByAmazon_Service_Interface
 */

require_once ('CheckoutByAmazon/Service/Interface.php');

/**
 * CheckoutByAmazon_Service_Client is an implementation of CheckoutByAmazon_Service
 *
 */
class CheckoutByAmazon_Service_Client implements CheckoutByAmazon_Service_Interface
{

    const SERVICE_VERSION = '2010-08-31';

    /** @var string */
    private  $_awsAccessKeyId = null;

    /** @var string */
    private  $_awsSecretAccessKey = null;

    /** @var array */
    private  $_config = array ('ServiceURL' => 'https://payments.amazon.de/cba/api/purchasecontract/',
                               'UserAgent' => 'CheckoutByAmazon Service PHP5 Library',
                               'SignatureVersion' => 2,
                               'SignatureMethod' => 'HmacSHA256',
                               'ProxyHost' => null,
                               'ProxyPort' => -1,
                               'MaxErrorRetry' => 3
                               );

    /**
     * Construct new Client
     *
     * @param string $awsAccessKeyId AWS Access Key ID
     * @param string $awsSecretAccessKey AWS Secret Access Key
     * @param array $config configuration options.
     * Valid configuration options are:
     * <ul>
     * <li>ServiceURL</li>
     * <li>UserAgent</li>
     * <li>SignatureVersion</li>
     * <li>TimesRetryOnError</li>
     * <li>ProxyHost</li>
     * <li>ProxyPort</li>
     * <li>MaxErrorRetry</li>
     * </ul>
     */
    public function __construct($awsAccessKeyId, $awsSecretAccessKey, $config = null)
    {
        iconv_set_encoding('output_encoding', 'UTF-8');
        iconv_set_encoding('input_encoding', 'UTF-8');
        iconv_set_encoding('internal_encoding', 'UTF-8');

        $this->_awsAccessKeyId = $awsAccessKeyId;
        $this->_awsSecretAccessKey = $awsSecretAccessKey;
        if (!is_null($config)) $this->_config = array_merge($this->_config, $config);
    }

    // Public API ------------------------------------------------------------//



    /**
     * Create Purchase Contract
     * @param mixed $request array of parameters for CheckoutByAmazon_Service_Model_CreatePurchaseContractRequest request
     * or CheckoutByAmazon_Service_Model_CreatePurchaseContractRequest object itself
     * @see CheckoutByAmazon_Service_Model_CreatePurchaseContract
     * @return CheckoutByAmazon_Service_Model_CreatePurchaseContractResponse CheckoutByAmazon_Service_Model_CreatePurchaseContractResponse
     *
     * @throws CheckoutByAmazon_Service_Exception
     */
    public function createPurchaseContract($request)
    {
        if (!$request instanceof CheckoutByAmazon_Service_Model_CreatePurchaseContractRequest) {
            require_once ('CheckoutByAmazon/Service/Model/CreatePurchaseContractRequest.php');
            $request = new CheckoutByAmazon_Service_Model_CreatePurchaseContractRequest($request);
        }
        require_once ('CheckoutByAmazon/Service/Model/CreatePurchaseContractResponse.php');
        return CheckoutByAmazon_Service_Model_CreatePurchaseContractResponse::fromXML($this->_invoke($this->_convertCreatePurchaseContract($request)));
    }



    /**
     * Get Purchase Contract
     * @param mixed $request array of parameters for CheckoutByAmazon_Service_Model_GetPurchaseContractRequest request
     * or CheckoutByAmazon_Service_Model_GetPurchaseContractRequest object itself
     * @see CheckoutByAmazon_Service_Model_GetPurchaseContract
     * @return CheckoutByAmazon_Service_Model_GetPurchaseContractResponse CheckoutByAmazon_Service_Model_GetPurchaseContractResponse
     *
     * @throws CheckoutByAmazon_Service_Exception
     */
    public function getPurchaseContract($request)
    {
        if (!$request instanceof CheckoutByAmazon_Service_Model_GetPurchaseContractRequest) {
            require_once ('CheckoutByAmazon/Service/Model/GetPurchaseContractRequest.php');
            $request = new CheckoutByAmazon_Service_Model_GetPurchaseContractRequest($request);
        }
        require_once ('CheckoutByAmazon/Service/Model/GetPurchaseContractResponse.php');
        return CheckoutByAmazon_Service_Model_GetPurchaseContractResponse::fromXML($this->_invoke($this->_convertGetPurchaseContract($request)));
    }



    /**
     * Set Purchase Items
     * @param mixed $request array of parameters for CheckoutByAmazon_Service_Model_SetPurchaseItemsRequest request
     * or CheckoutByAmazon_Service_Model_SetPurchaseItemsRequest object itself
     * @see CheckoutByAmazon_Service_Model_SetPurchaseItems
     * @return CheckoutByAmazon_Service_Model_SetPurchaseItemsResponse CheckoutByAmazon_Service_Model_SetPurchaseItemsResponse
     *
     * @throws CheckoutByAmazon_Service_Exception
     */
    public function setPurchaseItems($request)
    {
        if (!$request instanceof CheckoutByAmazon_Service_Model_SetPurchaseItemsRequest) {
            require_once ('CheckoutByAmazon/Service/Model/SetPurchaseItemsRequest.php');
            $request = new CheckoutByAmazon_Service_Model_SetPurchaseItemsRequest($request);
        }
        require_once ('CheckoutByAmazon/Service/Model/SetPurchaseItemsResponse.php');
        return CheckoutByAmazon_Service_Model_SetPurchaseItemsResponse::fromXML($this->_invoke($this->_convertSetPurchaseItems($request)));
    }



    /**
     * Complete Purchase Contract
     * @param mixed $request array of parameters for CheckoutByAmazon_Service_Model_CompletePurchaseContractRequest request
     * or CheckoutByAmazon_Service_Model_CompletePurchaseContractRequest object itself
     * @see CheckoutByAmazon_Service_Model_CompletePurchaseContract
     * @return CheckoutByAmazon_Service_Model_CompletePurchaseContractResponse CheckoutByAmazon_Service_Model_CompletePurchaseContractResponse
     *
     * @throws CheckoutByAmazon_Service_Exception
     */
    public function completePurchaseContract($request)
    {
        if (!$request instanceof CheckoutByAmazon_Service_Model_CompletePurchaseContractRequest) {
            require_once ('CheckoutByAmazon/Service/Model/CompletePurchaseContractRequest.php');
            $request = new CheckoutByAmazon_Service_Model_CompletePurchaseContractRequest($request);
        }
        require_once ('CheckoutByAmazon/Service/Model/CompletePurchaseContractResponse.php');
        return CheckoutByAmazon_Service_Model_CompletePurchaseContractResponse::fromXML($this->_invoke($this->_convertCompletePurchaseContract($request)));
    }



    /**
     * Set Contract Charges
     * @param mixed $request array of parameters for CheckoutByAmazon_Service_Model_SetContractChargesRequest request
     * or CheckoutByAmazon_Service_Model_SetContractChargesRequest object itself
     * @see CheckoutByAmazon_Service_Model_SetContractCharges
     * @return CheckoutByAmazon_Service_Model_SetContractChargesResponse CheckoutByAmazon_Service_Model_SetContractChargesResponse
     *
     * @throws CheckoutByAmazon_Service_Exception
     */
    public function setContractCharges($request)
    {
        if (!$request instanceof CheckoutByAmazon_Service_Model_SetContractChargesRequest) {
            require_once ('CheckoutByAmazon/Service/Model/SetContractChargesRequest.php');
            $request = new CheckoutByAmazon_Service_Model_SetContractChargesRequest($request);
        }
        require_once ('CheckoutByAmazon/Service/Model/SetContractChargesResponse.php');

        return CheckoutByAmazon_Service_Model_SetContractChargesResponse::fromXML($this->_invoke($this->_convertSetContractCharges($request)));
    }

        // Private API ------------------------------------------------------------//

    /**
     * Invoke request and return response
     */
    private function _invoke(array $parameters)
    {
        $actionName = $parameters["Action"];
        $response = array();
        $responseBody = null;
        $statusCode = 200;

        /* Submit the request and read response body */
        try {

            /* Add required request parameters */
            $parameters = $this->_addRequiredParameters($parameters);

            $shouldRetry = true;
            $retries = 0;
            do {
                try {
                        $response = $this->_httpPost($parameters);
                        if ($response['Status'] === 200) {
                            $shouldRetry = false;
                        } else {
                            if ($response['Status'] === 500 || $response['Status'] === 503) {
                                $shouldRetry = true;
                                $this->_pauseOnRetry(++$retries, $response['Status']);
                            } else {
                                throw $this->_reportAnyErrors($response['ResponseBody'], $response['Status']);
                            }
                       }
                /* Rethrow on deserializer error */
                } catch (Exception $e) {
                    require_once ('CheckoutByAmazon/Service/Exception.php');
                    if ($e instanceof CheckoutByAmazon_Service_Exception) {
                        throw $e;
                    } else {
                        require_once ('CheckoutByAmazon/Service/Exception.php');
                        throw new CheckoutByAmazon_Service_Exception(array('Exception' => $e, 'Message' => $e->getMessage()));
                    }
                }

            } while ($shouldRetry);

        } catch (CheckoutByAmazon_Service_Exception $se) {
            throw $se;
        } catch (Exception $t) {
            throw new CheckoutByAmazon_Service_Exception(array('Exception' => $t, 'Message' => $t->getMessage()));
        }

        return $response['ResponseBody'];
    }

    /**
     * Look for additional error strings in the response and return formatted exception
     */
    private function _reportAnyErrors($responseBody, $status, Exception $e =  null)
    {
        $ex = null;
        if (!is_null($responseBody) && strpos($responseBody, '<') === 0) {
            if (preg_match('@<RequestId>(.*)</RequestId>.*<Error><Code>(.*)</Code><Message>(.*)</Message></Error>.*(<Error>)?@mi',
                $responseBody, $errorMatcherOne)) {

                $requestId = $errorMatcherOne[1];
                $code = $errorMatcherOne[2];
                $message = $errorMatcherOne[3];

                require_once ('CheckoutByAmazon/Service/Exception.php');
                $ex = new CheckoutByAmazon_Service_Exception(array ('Message' => $message, 'StatusCode' => $status, 'ErrorCode' => $code,
                                                           'ErrorType' => 'Unknown', 'RequestId' => $requestId, 'XML' => $responseBody));

            } elseif (preg_match('@<Error><Code>(.*)</Code><Message>(.*)</Message></Error>.*(<Error>)?.*<RequestID>(.*)</RequestID>@mi',
                $responseBody, $errorMatcherTwo)) {

                $code = $errorMatcherTwo[1];
                $message = $errorMatcherTwo[2];
                $requestId = $errorMatcherTwo[4];
                require_once ('CheckoutByAmazon/Service/Exception.php');
                $ex = new CheckoutByAmazon_Service_Exception(array ('Message' => $message, 'StatusCode' => $status, 'ErrorCode' => $code,
                                                              'ErrorType' => 'Unknown', 'RequestId' => $requestId, 'XML' => $responseBody));
            } elseif (preg_match('@<Error>.*<Type>(.*)</Type>.*<Code>(.*)</Code>.*<Message>(.*)</Message>.*</Error>.*(<Error>)?.*<RequestId>(.*)</RequestId>@mis',
                $responseBody, $errorMatcherThree)) {

                $type = $errorMatcherThree[1];
                $code = $errorMatcherThree[2];
                $message = $errorMatcherThree[3];
                $requestId = $errorMatcherThree[5];
                require_once ('CheckoutByAmazon/Service/Exception.php');
                $ex = new CheckoutByAmazon_Service_Exception(array ('Message' => $message, 'StatusCode' => $status, 'ErrorCode' => $code,
                                                              'ErrorType' => $type, 'RequestId' => $requestId, 'XML' => $responseBody));

            } else {
                require_once ('CheckoutByAmazon/Service/Exception.php');
                $ex = new CheckoutByAmazon_Service_Exception(array('Message' => 'Internal Error', 'StatusCode' => $status));
            }
        } else {
            require_once ('CheckoutByAmazon/Service/Exception.php');
            $ex = new CheckoutByAmazon_Service_Exception(array('Message' => 'Internal Error', 'StatusCode' => $status));
        }
        return $ex;
    }



    /**
     * Perform HTTP post with exponential retries on error 500 and 503
     *
     */
    private function _httpPost(array $parameters)
    {
    	$CBAServiceEndpoint = $this->_config['ServiceURL'];
    	// $query = http_build_query($parameters);
		$query = $this->_getParametersAsString($parameters);
    	//initialize CURL
    	$curlHandle = curl_init();

    	//compose CURL request
    	curl_setopt($curlHandle, CURLOPT_URL, $CBAServiceEndpoint);
    	curl_setopt($curlHandle, CURLOPT_USERAGENT, $this->_config['UserAgent']);
    	curl_setopt($curlHandle, CURLOPT_POST, true);
    	curl_setopt($curlHandle, CURLOPT_POSTFIELDS, $query);
    	curl_setopt($curlHandle, CURLOPT_FRESH_CONNECT, true);
    	curl_setopt($curlHandle, CURLOPT_SSL_VERIFYPEER, true);
    	curl_setopt($curlHandle, CURLOPT_SSL_VERIFYHOST, 2);
    	curl_setopt($curlHandle, CURLOPT_CAINFO, 'CheckoutByAmazon/ca-bundle.crt');
    	curl_setopt($curlHandle, CURLOPT_CAPATH, 'CheckoutByAmazon/ca-bundle.crt');
    	curl_setopt($curlHandle, CURLOPT_FOLLOWLOCATION, false);
    	curl_setopt($curlHandle, CURLOPT_MAXREDIRS, 0);
    	curl_setopt($curlHandle, CURLOPT_HEADER, true);
    	curl_setopt($curlHandle, CURLOPT_RETURNTRANSFER, true);
    	curl_setopt($curlHandle, CURLOPT_NOSIGNAL, true);
    	curl_setopt($curlHandle, CURLOPT_HEADER, false);
    	// Execute the request
    	$responseBody = curl_exec($curlHandle);
    	$info = curl_getinfo($curlHandle);
    	// to grab the response code only from the Header
    	$code = $info["http_code"];

    	//close the CURL conection
    	curl_close($curlHandle);



    	return array ('Status' => (int)$code, 'ResponseBody' => $responseBody);
    }

    /**
     * Exponential sleep on failed request
     * @param retries current retry
     * @throws CheckoutByAmazon_Service_Exception if maximum number of retries has been reached
     */
    private function _pauseOnRetry($retries, $status)
    {
        if ($retries <= $this->_config['MaxErrorRetry']) {
            $delay = (int) (pow(4, $retries) * 100000) ;
            usleep($delay);
        } else {
            require_once ('CheckoutByAmazon/Service/Exception.php');
            throw new CheckoutByAmazon_Service_Exception (array ('Message' => "Maximum number of retry attempts reached :  $retries", 'StatusCode' => $status));
        }
    }

    /**
     * Add authentication related and version parameters
     */
    private function _addRequiredParameters(array $parameters)
    {
        $parameters['AWSAccessKeyId'] = $this->_awsAccessKeyId;
        $parameters['Timestamp'] = $this->_getFormattedTimestamp();
        $parameters['Version'] = self::SERVICE_VERSION;
        $parameters['SignatureVersion'] = $this->_config['SignatureVersion'];
        if ($parameters['SignatureVersion'] > 1) {
            $parameters['SignatureMethod'] = $this->_config['SignatureMethod'];
        }
        $parameters['Signature'] = $this->_signParameters($parameters, $this->_awsSecretAccessKey);

        return $parameters;
    }

    /**
     * Convert paremeters to Url encoded query string
     */
    private function _getParametersAsString(array $parameters)
    {
        $queryParameters = array();
        foreach ($parameters as $key => $value) {
            $queryParameters[] = $key . '=' . $this->_urlencode($value);
        }
        return implode('&', $queryParameters);
    }


    /**
     * Computes RFC 2104-compliant HMAC signature for request parameters
     * Implements AWS Signature, as per following spec:
     *
     * If Signature Version is 0, it signs concatenated Action and Timestamp
     *
     * If Signature Version is 1, it performs the following:
     *
     * Sorts all  parameters (including SignatureVersion and excluding Signature,
     * the value of which is being created), ignoring case.
     *
     * Iterate over the sorted list and append the parameter name (in original case)
     * and then its value. It will not URL-encode the parameter values before
     * constructing this string. There are no separators.
     *
     * If Signature Version is 2, string to sign is based on following:
     *
     *    1. The HTTP Request Method followed by an ASCII newline (%0A)
     *    2. The HTTP Host header in the form of lowercase host, followed by an ASCII newline.
     *    3. The URL encoded HTTP absolute path component of the URI
     *       (up to but not including the query string parameters);
     *       if this is empty use a forward '/'. This parameter is followed by an ASCII newline.
     *    4. The concatenation of all query string components (names and values)
     *       as UTF-8 characters which are URL encoded as per RFC 3986
     *       (hex characters MUST be uppercase), sorted using lexicographic byte ordering.
     *       Parameter names are separated from their values by the '=' character
     *       (ASCII character 61), even if the value is empty.
     *       Pairs of parameter and values are separated by the '&' character (ASCII code 38).
     *
     */
    private function _signParameters(array $parameters, $key) {
        $signatureVersion = $parameters['SignatureVersion'];
        $algorithm = "HmacSHA1";
        $stringToSign = null;
        if (0 === $signatureVersion) {
            $stringToSign = $this->_calculateStringToSignV0($parameters);
        } else if (1 === $signatureVersion) {
            $stringToSign = $this->_calculateStringToSignV1($parameters);
        } else if (2 === $signatureVersion) {
            $algorithm = $this->_config['SignatureMethod'];
            $parameters['SignatureMethod'] = $algorithm;
            $stringToSign = $this->_calculateStringToSignV2($parameters);
        } else {
            throw new Exception("Invalid Signature Version specified");
        }
        return $this->_sign($stringToSign, $key, $algorithm);
    }

    /**
     * Calculate String to Sign for SignatureVersion 0
     * @param array $parameters request parameters
     * @return String to Sign
     */
    private function _calculateStringToSignV0(array $parameters) {
        return $parameters['Action'] .  $parameters['Timestamp'];
    }

    /**
     * Calculate String to Sign for SignatureVersion 1
     * @param array $parameters request parameters
     * @return String to Sign
     */
    private function _calculateStringToSignV1(array $parameters) {
        $data = '';
        uksort($parameters, 'strcasecmp');
        foreach ($parameters as $parameterName => $parameterValue) {
            $data .= $parameterName . $parameterValue;
        }
        return $data;
    }

    /**
     * Calculate String to Sign for SignatureVersion 2
     * @param array $parameters request parameters
     * @return String to Sign
     */
    private function _calculateStringToSignV2(array $parameters) {
        $data = 'POST';
        $data .= "\n";
        $endpoint = parse_url ($this->_config['ServiceURL']);
        $data .= $endpoint['host'];
        $data .= "\n";
        $uri = array_key_exists('path', $endpoint) ? $endpoint['path'] : null;
        if (!isset ($uri)) {
        	$uri = "/";
        }
		$uriencoded = implode("/", array_map(array($this, "_urlencode"), explode("/", $uri)));
        $data .= $uriencoded;
        $data .= "\n";
        uksort($parameters, 'strcmp');
        $data .= $this->_getParametersAsString($parameters);
        return $data;
    }

    private function _urlencode($value) {
		return str_replace('%7E', '~', rawurlencode($value));
    }


    /**
     * Computes RFC 2104-compliant HMAC signature.
     */
    private function _sign($data, $key, $algorithm)
    {
        if ($algorithm === 'HmacSHA1') {
            $hash = 'sha1';
        } else if ($algorithm === 'HmacSHA256') {
            $hash = 'sha256';
        } else {
            throw new Exception ("Non-supported signing method specified");
        }
        return base64_encode(
            hash_hmac($hash, $data, $key, true)
        );
    }


    /**
     * Formats date as ISO 8601 timestamp
     */
    private function _getFormattedTimestamp()
    {
        return gmdate("Y-m-d\TH:i:s.\\0\\0\\0\\Z", time());
    }



    /**
     * Convert CreatePurchaseContractRequest to name value pairs
     */
    private function _convertCreatePurchaseContract($request) {

        $parameters = array();
        $parameters['Action'] = 'CreatePurchaseContract';
        if ($request->isSetPurchaseContractMetadata()) {
            $parameters['PurchaseContractMetadata'] =  $request->getPurchaseContractMetadata();
        }

        return $parameters;
    }


    /**
     * Convert GetPurchaseContractRequest to name value pairs
     */
    private function _convertGetPurchaseContract($request) {

        $parameters = array();
        $parameters['Action'] = 'GetPurchaseContract';
        if ($request->isSetPurchaseContractId()) {
            $parameters['PurchaseContractId'] =  $request->getPurchaseContractId();
        }

        return $parameters;
    }


    /**
     * Convert SetPurchaseItemsRequest to name value pairs
     */
    private function _convertSetPurchaseItems($request) {

        $parameters = array();
        $parameters['Action'] = 'SetPurchaseItems';
        if ($request->isSetPurchaseContractId()) {
            $parameters['PurchaseContractId'] =  $request->getPurchaseContractId();
        }
        if ($request->isSetPurchaseItems()) {
            $purchaseItemssetPurchaseItemsRequest = $request->getPurchaseItems();
            foreach ($purchaseItemssetPurchaseItemsRequest->getPurchaseItem() as $purchaseItempurchaseItemsIndex => $purchaseItempurchaseItems) {
                if ($purchaseItempurchaseItems->isSetMerchantItemId()) {
                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'MerchantItemId'] =  $purchaseItempurchaseItems->getMerchantItemId();
                }
                if ($purchaseItempurchaseItems->isSetSKU()) {
                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'SKU'] =  $purchaseItempurchaseItems->getSKU();
                }
                if ($purchaseItempurchaseItems->isSetMerchantId()) {
                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'MerchantId'] =  $purchaseItempurchaseItems->getMerchantId();
                }
                if ($purchaseItempurchaseItems->isSetTitle()) {
                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'Title'] =  $purchaseItempurchaseItems->getTitle();
                }
                if ($purchaseItempurchaseItems->isSetDescription()) {
                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'Description'] =  $purchaseItempurchaseItems->getDescription();
                }
                if ($purchaseItempurchaseItems->isSetUnitPrice()) {
                    $unitPricepurchaseItem = $purchaseItempurchaseItems->getUnitPrice();
                    if ($unitPricepurchaseItem->isSetAmount()) {
                        $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'UnitPrice' . '.' . 'Amount'] =  $unitPricepurchaseItem->getAmount();
                    }
                    if ($unitPricepurchaseItem->isSetCurrencyCode()) {
                        $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'UnitPrice' . '.' . 'CurrencyCode'] =  $unitPricepurchaseItem->getCurrencyCode();
                    }
                }
                if ($purchaseItempurchaseItems->isSetQuantity()) {
                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'Quantity'] =  $purchaseItempurchaseItems->getQuantity();
                }
                if ($purchaseItempurchaseItems->isSetURL()) {
                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'URL'] =  $purchaseItempurchaseItems->getURL();
                }
                if ($purchaseItempurchaseItems->isSetCategory()) {
                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'Category'] =  $purchaseItempurchaseItems->getCategory();
                }
                if ($purchaseItempurchaseItems->isSetFulfillmentNetwork()) {
                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'FulfillmentNetwork'] =  $purchaseItempurchaseItems->getFulfillmentNetwork();
                }
                if ($purchaseItempurchaseItems->isSetItemCustomData()) {
                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'ItemCustomData'] =  $purchaseItempurchaseItems->getItemCustomData();
                }
                if ($purchaseItempurchaseItems->isSetProductType()) {
                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'ProductType'] =  $purchaseItempurchaseItems->getProductType();
                }
                if ($purchaseItempurchaseItems->isSetPhysicalProductAttributes()) {
                    $physicalProductAttributespurchaseItem = $purchaseItempurchaseItems->getPhysicalProductAttributes();
                    if ($physicalProductAttributespurchaseItem->isSetWeight()) {
                        $weightphysicalProductAttributes = $physicalProductAttributespurchaseItem->getWeight();
                        if ($weightphysicalProductAttributes->isSetValue()) {
                            $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'Weight' . '.' . 'Value'] =  $weightphysicalProductAttributes->getValue();
                        }
                        if ($weightphysicalProductAttributes->isSetUnit()) {
                            $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'Weight' . '.' . 'Unit'] =  $weightphysicalProductAttributes->getUnit();
                        }
                    }
                    if ($physicalProductAttributespurchaseItem->isSetCondition()) {
                        $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'Condition'] =  $physicalProductAttributespurchaseItem->getCondition();
                    }
                    if ($physicalProductAttributespurchaseItem->isSetGiftOption()) {
                        $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'GiftOption'] =  $physicalProductAttributespurchaseItem->getGiftOption();
                    }
                    if ($physicalProductAttributespurchaseItem->isSetGiftMessage()) {
                        $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'GiftMessage'] =  $physicalProductAttributespurchaseItem->getGiftMessage();
                    }
                    if ($physicalProductAttributespurchaseItem->isSetDeliveryMethod()) {
                        $deliveryMethodphysicalProductAttributes = $physicalProductAttributespurchaseItem->getDeliveryMethod();
                        if ($deliveryMethodphysicalProductAttributes->isSetServiceLevel()) {
                            $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'DeliveryMethod' . '.' . 'ServiceLevel'] =  $deliveryMethodphysicalProductAttributes->getServiceLevel();
                        }
                        if ($deliveryMethodphysicalProductAttributes->isSetDisplayableShippingLabel()) {
                            $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'DeliveryMethod' . '.' . 'DisplayableShippingLabel'] =  $deliveryMethodphysicalProductAttributes->getDisplayableShippingLabel();
                        }
                        if ($deliveryMethodphysicalProductAttributes->isSetDestinationName()) {
                            $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'DeliveryMethod' . '.' . 'DestinationName'] =  $deliveryMethodphysicalProductAttributes->getDestinationName();
                        }
                        if ($deliveryMethodphysicalProductAttributes->isSetShippingCustomData()) {
                            $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'DeliveryMethod' . '.' . 'ShippingCustomData'] =  $deliveryMethodphysicalProductAttributes->getShippingCustomData();
                        }
                    }
                    if ($physicalProductAttributespurchaseItem->isSetItemCharges()) {
                        $itemChargesphysicalProductAttributes = $physicalProductAttributespurchaseItem->getItemCharges();
                        if ($itemChargesphysicalProductAttributes->isSetTax()) {
                            $taxitemCharges = $itemChargesphysicalProductAttributes->getTax();
                            if ($taxitemCharges->isSetAmount()) {
                                $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'ItemCharges' . '.' . 'Tax' . '.' . 'Amount'] =  $taxitemCharges->getAmount();
                            }
                            if ($taxitemCharges->isSetCurrencyCode()) {
                                $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'ItemCharges' . '.' . 'Tax' . '.' . 'CurrencyCode'] =  $taxitemCharges->getCurrencyCode();
                            }
                        }
                        if ($itemChargesphysicalProductAttributes->isSetShipping()) {
                            $shippingitemCharges = $itemChargesphysicalProductAttributes->getShipping();
                            if ($shippingitemCharges->isSetAmount()) {
                                $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'ItemCharges' . '.' . 'Shipping' . '.' . 'Amount'] =  $shippingitemCharges->getAmount();
                            }
                            if ($shippingitemCharges->isSetCurrencyCode()) {
                                $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'ItemCharges' . '.' . 'Shipping' . '.' . 'CurrencyCode'] =  $shippingitemCharges->getCurrencyCode();
                            }
                        }
                        if ($itemChargesphysicalProductAttributes->isSetGiftWrap()) {
                            $giftWrapitemCharges = $itemChargesphysicalProductAttributes->getGiftWrap();
                            if ($giftWrapitemCharges->isSetAmount()) {
                                $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'ItemCharges' . '.' . 'GiftWrap' . '.' . 'Amount'] =  $giftWrapitemCharges->getAmount();
                            }
                            if ($giftWrapitemCharges->isSetCurrencyCode()) {
                                $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'ItemCharges' . '.' . 'GiftWrap' . '.' . 'CurrencyCode'] =  $giftWrapitemCharges->getCurrencyCode();
                            }
                        }
                        if ($itemChargesphysicalProductAttributes->isSetPromotions()) {
                            $promotionsitemCharges = $itemChargesphysicalProductAttributes->getPromotions();
                            foreach ($promotionsitemCharges->getPromotion() as $promotionpromotionsIndex => $promotionpromotions) {
                                if ($promotionpromotions->isSetPromotionId()) {
                                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'ItemCharges' . '.' . 'Promotions' . '.' . 'Promotion' . '.'  . ($promotionpromotionsIndex + 1) . '.' . 'PromotionId'] =  $promotionpromotions->getPromotionId();
                                }
                                if ($promotionpromotions->isSetDescription()) {
                                    $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'ItemCharges' . '.' . 'Promotions' . '.' . 'Promotion' . '.'  . ($promotionpromotionsIndex + 1) . '.' . 'Description'] =  $promotionpromotions->getDescription();
                                }
                                if ($promotionpromotions->isSetDiscount()) {
                                    $discountpromotion = $promotionpromotions->getDiscount();
                                    if ($discountpromotion->isSetAmount()) {
                                        $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'ItemCharges' . '.' . 'Promotions' . '.' . 'Promotion' . '.'  . ($promotionpromotionsIndex + 1) . '.' . 'Discount' . '.' . 'Amount'] =  $discountpromotion->getAmount();
                                    }
                                    if ($discountpromotion->isSetCurrencyCode()) {
                                        $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'PhysicalProductAttributes' . '.' . 'ItemCharges' . '.' . 'Promotions' . '.' . 'Promotion' . '.'  . ($promotionpromotionsIndex + 1) . '.' . 'Discount' . '.' . 'CurrencyCode'] =  $discountpromotion->getCurrencyCode();
                                    }
                                }

                            }
                        }
                    }
                }
                if ($purchaseItempurchaseItems->isSetDigitalProductAttributes()) {
                    $digitalProductAttributespurchaseItem = $purchaseItempurchaseItems->getDigitalProductAttributes();
                    if ($digitalProductAttributespurchaseItem->isSetdummyDigitalProperty()) {
                        $parameters['PurchaseItems' . '.' . 'PurchaseItem' . '.'  . ($purchaseItempurchaseItemsIndex + 1) . '.' . 'DigitalProductAttributes' . '.' . 'dummyDigitalProperty'] =  $digitalProductAttributespurchaseItem->getdummyDigitalProperty();
                    }
                }

            }
        }

        return $parameters;
    }


    /**
     * Convert CompletePurchaseContractRequest to name value pairs
     */
    private function _convertCompletePurchaseContract($request) {

        $parameters = array();
        $parameters['Action'] = 'CompletePurchaseContract';
        if ($request->isSetPurchaseContractId()) {
            $parameters['PurchaseContractId'] =  $request->getPurchaseContractId();
        }
        if ($request->isSetIntegratorId()) {
            $parameters['IntegratorId'] =  $request->getIntegratorId();
        }
        if ($request->isSetIntegratorName()) {
            $parameters['IntegratorName'] =  $request->getIntegratorName();
        }
        if ($request->isSetInstantOrderProcessingNotificationURLs()) {
            $instantOrderProcessingNotificationURLscompletePurchaseContractRequest = $request->getInstantOrderProcessingNotificationURLs();
            if ($instantOrderProcessingNotificationURLscompletePurchaseContractRequest->isSetIntegratorURL()) {
                $parameters['InstantOrderProcessingNotificationURLs' . '.' . 'IntegratorURL'] =  $instantOrderProcessingNotificationURLscompletePurchaseContractRequest->getIntegratorURL();
            }
            if ($instantOrderProcessingNotificationURLscompletePurchaseContractRequest->isSetMerchantURL()) {
                $parameters['InstantOrderProcessingNotificationURLs' . '.' . 'MerchantURL'] =  $instantOrderProcessingNotificationURLscompletePurchaseContractRequest->getMerchantURL();
            }
        }

        return $parameters;
    }


    /**
     * Convert SetContractChargesRequest to name value pairs
     */
    private function _convertSetContractCharges($request) {

        $parameters = array();
        $parameters['Action'] = 'SetContractCharges';
        if ($request->isSetPurchaseContractId()) {
            $parameters['PurchaseContractId'] =  $request->getPurchaseContractId();
        }
        if ($request->isSetCharges()) {
            $chargessetContractChargesRequest = $request->getCharges();
            if ($chargessetContractChargesRequest->isSetTax()) {
                $taxcharges = $chargessetContractChargesRequest->getTax();
                if ($taxcharges->isSetAmount()) {
                    $parameters['Charges' . '.' . 'Tax' . '.' . 'Amount'] =  $taxcharges->getAmount();
                }
                if ($taxcharges->isSetCurrencyCode()) {
                    $parameters['Charges' . '.' . 'Tax' . '.' . 'CurrencyCode'] =  $taxcharges->getCurrencyCode();
                }
            }
            if ($chargessetContractChargesRequest->isSetShipping()) {
                $shippingcharges = $chargessetContractChargesRequest->getShipping();
                if ($shippingcharges->isSetAmount()) {
                    $parameters['Charges' . '.' . 'Shipping' . '.' . 'Amount'] =  $shippingcharges->getAmount();
                }
                if ($shippingcharges->isSetCurrencyCode()) {
                    $parameters['Charges' . '.' . 'Shipping' . '.' . 'CurrencyCode'] =  $shippingcharges->getCurrencyCode();
                }
            }
            if ($chargessetContractChargesRequest->isSetGiftWrap()) {
                $giftWrapcharges = $chargessetContractChargesRequest->getGiftWrap();
                if ($giftWrapcharges->isSetAmount()) {
                    $parameters['Charges' . '.' . 'GiftWrap' . '.' . 'Amount'] =  $giftWrapcharges->getAmount();
                }
                if ($giftWrapcharges->isSetCurrencyCode()) {
                    $parameters['Charges' . '.' . 'GiftWrap' . '.' . 'CurrencyCode'] =  $giftWrapcharges->getCurrencyCode();
                }
            }
            if ($chargessetContractChargesRequest->isSetPromotions()) {
                $promotionscharges = $chargessetContractChargesRequest->getPromotions();
                foreach ($promotionscharges->getPromotion() as $promotionpromotionsIndex => $promotionpromotions) {
                    if ($promotionpromotions->isSetPromotionId()) {
                        $parameters['Charges' . '.' . 'Promotions' . '.' . 'Promotion' . '.'  . ($promotionpromotionsIndex + 1) . '.' . 'PromotionId'] =  $promotionpromotions->getPromotionId();
                    }
                    if ($promotionpromotions->isSetDescription()) {
                        $parameters['Charges' . '.' . 'Promotions' . '.' . 'Promotion' . '.'  . ($promotionpromotionsIndex + 1) . '.' . 'Description'] =  $promotionpromotions->getDescription();
                    }
                    if ($promotionpromotions->isSetDiscount()) {
                        $discountpromotion = $promotionpromotions->getDiscount();
                        if ($discountpromotion->isSetAmount()) {
                            $parameters['Charges' . '.' . 'Promotions' . '.' . 'Promotion' . '.'  . ($promotionpromotionsIndex + 1) . '.' . 'Discount' . '.' . 'Amount'] =  $discountpromotion->getAmount();
                        }
                        if ($discountpromotion->isSetCurrencyCode()) {
                            $parameters['Charges' . '.' . 'Promotions' . '.' . 'Promotion' . '.'  . ($promotionpromotionsIndex + 1) . '.' . 'Discount' . '.' . 'CurrencyCode'] =  $discountpromotion->getCurrencyCode();
                        }
                    }

                }
            }
        }

        return $parameters;
    }


}
