<?php
/**
 * +----------------------------------------------------------+
 * Ersetzt die Platzhalter in der CSS-Datei durch gesetzte
 * Werte.
 *
 * ------ HOWTO -------
 *
 * "/*=" leitet die "schlauen" Kommentare ein 
 * 
 * -- Definitionen --s
 * @define myvalue 15;
 * @redefine test [myvalue + 5];
 *
 * -- Anzahl der Nachkommastellen bei Berechnungen --
 * @set precision 3;
 *
 * -- Berechnet das Raster --
 * @set grid colnum colwidth gutter (optional: gridname);
 * Folgende Variablen werden dabei automatisch definiert: grid_pagewidth, grid_colspan1 .. grid_colspanN, grid_gutter, grid_colwidth, grid_colnum
 *
 * -- Berechnet das Layout --
 * @set layout grid1 1 2 1; -> 1., 2., 3. Spalte
 * @set layout grid1 0 2 1; -> 2., 3. Spalte
 * Folgende Variablen werden dabei automatisch definiert: layoutXYZ_left (1 Spalte), layoutXYZ_main (2 Spalten), layoutXYZ_right (1 Spalte).
 * Sie enhalten die Breiten des Grids 'grid1'
 * 
 * -- Nach allen Variablen, die mit "grid_" anfangen, wird "px" angefügt --
 * @set append grid_* px;
 *
 * -- Einen Wert zu einer Variable addieren --
 * @add var [var2];
 * @add var 4;
 *
 * 
 * -- Einsatz der Ausgabe --
 * css{
 *    attribute: value /*= 0 0 [myvalue*2+test] *(kein Leerzeichen!)/; 
 * }
 * 
 * 
 * @name DynamicCSS
 * @package Classes
 * @subpackage Template
 * @since 
 * @version 0.94
 * @license GNU General Public License v3
 * @author Leonid Lezner <leonid.lezner@shopdienst.com>
 * +----------------------------------------------------------+
 */



/**
 * Ersetzt die Platzhalter in der CSS-Datei
 * durch zuvor gesetzte Werte.
 * 
 * @author Leonid Lezner <leonid.lezner@shopdienst.com>
 */
class CSS extends SourceMerger {
   
   // Variablen, die im CSS-Code gelten
   private $aVars = array();
   
   // Suffix-Liste
   private $aSuffixForVar = array();
   
   // Sollen die Kommentare gelöscht werden  
   private $bRemoveComments = true;
   
   // Sollen Imports gelöscht werden
   private $bRemoveImports = true;
   
   // Zeichen, welches den variable Kommentar einleitet
   private $sVarCommentStarter = "=";
   
   // Zeichen, welches den Befehl einleitet
   private $sVarCommandStarter = "@";  
   
   // Objekt zum Berechnen von Gleichungen
   private $oEvalMath = null;
   
   // Anzahl Nachkommastellen
   private $iPrecision = 3;
   
   
   
   /**
    * Konstruktor, setzt die Erweiterung auf "css" und lädt die Klasse EvalMath
    *
    * 
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   public function CSS($bUseCaching = true, $bUseVersioning = true){
      // Erweiterung setzen
      parent::__construct("css", $bUseCaching, $bUseVersioning);
      
      // Klasse zum Berechnen von Gleichungen initialisieren
      $this->oEvalMath = new EvalMath();
      
      // Keine Fehler ausgeben
      $this->oEvalMath->suppress_errors = true;
   }
   
   
   /**
    * Speichert die Dateinamen ab und prüft, ob Cache neu angelegt werden muss
    *
    *
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   public function addFiles($aFileList, $sFileName, $sSuffix = "", $sMedia = "screen, projection", $sConditionalComment = "", $bAlternate = false, $sTitle = ""){
      // Media
      if(!$sMedia)
         $sMedia = 'screen, projection';
         
      // Stylesheet Titel
      if($sTitle)
         $sTitle = 'title="'.$sTitle.'" ';
         
      // Als alternatives Stylesheet kennzeichnen
      if($bAlternate)
         $sAlternate = 'alternate ';
      else
         $sAlternate = ''; 
      
      // Link zum Stylesheet mit dem Platzhalter für die URL
      $sLink = '<link rel="'.$sAlternate.'stylesheet" type="text/css" href="%s" media="'.$sMedia.'" '.$sTitle.' />'."\n";
      
      // Conditional Comments für den IE
      if($sConditionalComment)
         $sLink = "<!--[if ".$sConditionalComment."]>\n".$sLink."<![endif]-->\n";
      
      // Dateien hinzufügen
      parent::addFiles($aFileList, $sFileName.$sSuffix, $sLink, $sSuffix);
   }
   
   
   /**
    * Bindet die Dateien ein und gibt ein <link>-Tag Zurück
    *
    * @param boolean bFullPathes Sind nur Dateinamen oder volle Pfade angegeben?
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   public function linkFiles($bFullPathes = false){
      // Link zu der Datei
      $sLink = "";
      
      // Dateipackages
      $aFilePackages = $this->loadFiles($bFullPathes);
      
      // aCustomInfo enthält den Format String mit dem Platzhalter für die URL
      foreach($aFilePackages as $oFilePackage){
         $sLink .= sprintf($oFilePackage->aCustomInfo, $oFilePackage->sFileUrl);
      }
      
      return trim($sLink, "\t");
   }
   
   
   /**
    * Setzt die Variable
    *
    *
    * @param string $sName Variablenname
    * @param string $sValue Variablenwert
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   public function setVar($sName, $sValue){
      // In Kleinbuchstaben wandeln
      $sName = strtolower($sName);
      
      // Variable setzen
      $this->aVars[$sName] = $sValue;
   }
   
   
   /**
    * Gibt die Variable zurück
    *
    *
    * @param string $sName Variablenname
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   public function getVar($sName){
      // In Kleinbuchstaben wandeln
      $sName = strtolower($sName);

      if(key_exists($sName, $this->aVars))
         return $this->aVars[$sName];
      else
         throw new Exception();
   }
   
   
   /**
    * Prüft, ob die Variable gesetzt ist
    *
    *
    * @param string $sName Variablenname
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   public function isVarSet($sName){
      // In Kleinbuchstaben wandeln
      $sName = strtolower($sName);
      
      if(key_exists($sName, $this->aVars))
         return true;
      else
         return false;
   }
   
   
   /**
    * Entfernt ungerwünschte Zeichen aus dem Code
    *
    *
    * @param string $sContent Dateiinhalt
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   protected function cleanSource($sContent){
      
      // Alle Kommentare löschen, bis auf die für die dynamisches CSS '/*='
      if($this->bRemoveComments){
         $sContent = preg_replace("#(/\*)(?!".$this->sVarCommentStarter.")(.*?)(\*/)#s", '', $sContent);
      }
      
      // Imports entfernen
      if($this->bRemoveImports){
         $sContent = preg_replace("#@import(.*);#", '', $sContent);
      }
      
      // Leerzeichen sinnvoll entfernen
      $aSpacesToRemove = array('{ ', ' }', ' {', '} ', ' ,', ', ', ': ', ' :', '; ', ' ;');
      
      foreach($aSpacesToRemove as $sSpace)
         $sContent = str_replace($sSpace, trim($sSpace), $sContent);
      
      // Zeilenumbrüche entfernen
      $sContent = str_replace(array("\r\n", "\r", "\n"), '', $sContent);

      return $sContent;
   }
   
   
   /**
    * Parst den Quellcode
    *
    *
    * @param string $sContent Dateiinhalt
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   protected function parseSource($sContent){
      // Neues Content
      $sNewContent = $sContent;
      
      // Suche nach Kommentaren mit Befehlen
      try{
         $sNewContent = preg_replace_callback("#/\*".$this->sVarCommentStarter."(.*?)\*/#si", array($this, 'extractCommentCommand'), $sNewContent);
      }catch (Exception $e){
         $sNewContent = $sContent;
      }
      
      // Suche nach Variablen-Kommentaren für Attribute und deren Werte
      try{
         $sNewContent = preg_replace_callback("#([A-Za-z-]+)\s*:((?:(?![;{}]).)+)/\*".$this->sVarCommentStarter."(.*?)\*/#si", 
                                                array($this, 'extractVarComment'), $sNewContent);
      }catch (Exception $e){
         $sNewContent = $sContent;
      }
      
      $sNewContent = str_replace("\t", '', $sNewContent);

      return $sNewContent;
   }
   
      
   /**
    * CALLBACK
    * Extrahiert aus dem CSS-Code die Kommentrare mit den Variablen
    *
    * "0 0 [varname+varname2] 0"
    *
    * @param array $aMatches
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   private function extractVarComment($aMatches, $bVarPartOnly = false){
      // Soll nur der variable Teil zurückgegeben werden?
      if(!$bVarPartOnly){
         // Attributname
         $sAttribute = $aMatches[1];
         
         // Der alte Wert
         $sOldValue = $aMatches[2];
         
         // Kommentar mit dem Ausdruck
         $sVarComment = $aMatches[3];
      }else{
         $sVarComment = $aMatches;
      }
      
      try{
         // Nach Variablen-Ausdrücken suchen
         $sVarComment = trim(preg_replace_callback("#\[(.*?)\](px|pt)*#si", array($this, 'calculateVariablePart'.($bVarPartOnly?'WithoutSuffix':'')), $sVarComment));
         
         // Soll nur der variable Teil zurückgegeben werden?
         if($bVarPartOnly){
            return $sVarComment;
         }else{
            // Attribut und Wert zusammensetzen
            return sprintf("%s:%s", trim($sAttribute), $sVarComment);
         }
      }catch(Exception $e) {
         // Soll nur der variable Teil zurückgegeben werden?
         if($bVarPartOnly){
            return 0;
         }else{
            // Bei Fehlern den alten Wert zurückgeben
            return sprintf("%s:%s", trim($sAttribute), trim($sOldValue));
         }
      }
   }
   


   /**
    * CALLBACK
    * Berechnet den Ausdruck aus dem Kommentar ohne Suffix
    *
    * "[varname+varname2]"
    *
    * @param array $aMatches
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   private function calculateVariablePartWithoutSuffix($aMatches){
      return $this->calculateVariablePart($aMatches, false);
   }



   /**
    * CALLBACK
    * Berechnet den Ausdruck aus dem Kommentar
    *
    * "[varname+varname2]"
    *
    * @param array $aMatches
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   private function calculateVariablePart($aMatches, $bAddSuffix = true){
      // Variabler Ausdruck
      $sVarPart = $aMatches[1];
      
      // Einheit
      if(isset($aMatches[2]))
         $sUnit = $aMatches[2];

      try{
         // Variablen durch ihre Werte ersetzen
         $sVarPartWithVals = preg_replace_callback("#[a-zA-Z_][a-zA-Z0-9_]*#", array($this, 'replaceVariableWithValue'), $sVarPart);

         // Es dürfen nur Zahlen und Zeichen wie +-*/(). und Leerzeichen stehen
         if(!preg_match('#[^(\d\+\-\*\(\)/\.\s)]#', $sVarPartWithVals, $aMatches)){
            // Es handelt sich um einen numerischen Ausdruck

            // Ausdruck berechnen
            $sVarPartWithVals = $this->oEvalMath->evaluate($sVarPartWithVals);
            
            // Prüfen, ob es bei der Berechnung Fehler gab
            if($this->oEvalMath->last_error != null)
               throw new Exception($this->oEvalMath->last_error);
            
            // Anzahl der Nachkommastellen muss positiv sein
            if($this->iPrecision < 0)
               $this->iPrecision = 0;
            
            // Auf x Nachkommastellen runden
            $sVarPartWithVals = round($sVarPartWithVals, $this->iPrecision);
            
            $sVarPartWithVals = str_replace(',', '.', $sVarPartWithVals);
            
            // Leerzeichen entfernen
            $sVarPartWithVals = str_replace(' ', '', $sVarPartWithVals);
         }
      }catch(Exception $e) {
         throw new Exception($e->getMessage());
      }
      
      // Die Einheit anfügen, wenn sie angegeben wurde
      if(isset($sUnit)){
         // Einige Einheiten erfordern nicht gerundete Werte
         if(($sUnit == "px" || $sUnit == "pt")){
            // Nachkommastellen abschneiden
            $sVarPartWithVals = floor($sVarPartWithVals);
         }
         $sVarPartWithVals .= $sUnit;
      }else{
         // Suffix anfügen, wenn Parameter bAddSuffix == true
         if($bAddSuffix){
            // Alle registrierten Suffixe durchsuchen
            foreach($this->aSuffixForVar as $sVarNameOfSuffixArray => $aAppend){
               // Variable aus dem Suffix-Array gefunden
               $bVarFound = false;
               
               // Variablen vergleichen. Wenn mind. eine passt, kann man den Suffix anhängen          
               if(preg_match_all("#[a-zA-Z_][a-zA-Z0-9_]*#si", $sVarPart, $aVarMatches)){
                  foreach($aVarMatches[0] as $sVarName){
                     // Variable vergleichen
                     if($aAppend[1]){
                        // Nur die n Zeichen des Namens (bis zum *) werden verglichen
                        if(substr($sVarName, 0, strlen($sVarNameOfSuffixArray)) == $sVarNameOfSuffixArray){
                           $bVarFound = true;
                           break;
                        }
                     }else{
                        // Der komplette Name wird verglichen
                        if($sVarName == $sVarNameOfSuffixArray){
                           $bVarFound = true;
                           break;
                        }
                     }
                  }
               }
   
               // Variable gefunden!
               if($bVarFound){
                  // Suffix
                  $sSuffix = $aAppend[0];
                  
                  // Suffix anhängen
                  $sVarPartWithVals .= $sSuffix;
                  
                  // Ein Suffix angefügt, aus der Schleife aussteigen
                  break;
               }
            }
         }
      }
      
      return $sVarPartWithVals;
   }
   
   
   /**
    * CALLBACK
    * Ersetzt die Variablen durch ihre Werte
    *
    * "varname", "varname2"
    *
    * @param array $aMatches
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   private function replaceVariableWithValue($aMatches){
      // Variablenname
      $sVarName = $aMatches[0];

      try{
         // Wert der Variable abfragen
         return $this->getVar($sVarName);
      }catch(Exception $e){
         throw new Exception('Variable '.$sVarName.' not found');
      }
   }
   
   
   /**
    * CALLBACK
    * Bearbeitet Kommentare mit Befehlen
    *
    * @param array $aMatches
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   private function extractCommentCommand($aMatches){
      // Alle Befehle
      $sCommands = $aMatches[1];

      // Suche nach allen Befehlen ("#befehl param1 param2;")
      if(preg_match_all("#".$this->sVarCommandStarter."([A-Za-z]+)\s*([a-zA-Z_][a-zA-Z0-9_]*)\s*(.*?);#si", $sCommands, $aSubMatches)){
         // Alle Befehle ausführen
         foreach($aSubMatches[1] as $iIndex => $sCommand){
            // Erstes Argument des Befehls
            $sParam1 = $aSubMatches[2][$iIndex];
            
            // Zweites Argument des Befehls
            $sParam2 = $aSubMatches[3][$iIndex];
            
            // Befehl ausführen
            $this->executeCommand(strtolower($sCommand), strtolower($sParam1), $sParam2);
         }
      }else{
         // Keine Befehle gefunden, nichts verändern
         return $aMatches[0];
      }
   }
   
   
   /**
    * Führt Befehle aus Kommentaren aus
    *
    * @param string $sCommand Befehl
    * @param string $sParam1 Parameter 1
    * @param string $sParam2 Parameter 2
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   private function executeCommand($sCommand, $sParam1, $sParam2){
      switch($sCommand){
         // Einen Wert auf eine Variable aufaddieren
         case 'add':
            // Den Wert der Zielvariable abfragen
            if($this->isVarSet($sParam1))
               $sOldValue = $this->getVar($sParam1);
            else
               $sOldValue = 0;
            
            // Zweiter Parameter leer, dann abbrechen
            if(!trim($sParam2))
               break;
            
            // Zweiten Parameter berechnen
            $sParam2 = $this->extractVarComment($sParam2, true);
            
            // Variable setzem
            $this->setVar($sParam1, intval($sOldValue)+intval($sParam2));
            
            break;
         
         // Variablen setzen
         case 'define':
            // Wenn die Variable existiert, wird der Befehl 'define' ignoriert
            if($this->isVarSet($sParam1))
               break;
         
         case 'redefine':
            // Wenn die Variable existiert, wir der Befehl 'redefine' diese überschreiben
            $sParam2 = $this->extractVarComment($sParam2, true);

            // Variable setzem
            $this->setVar($sParam1, $sParam2);
            
            break;
         
         // Konfiguration
         case 'set':
            // Anzahl der Nachkommastellen
            if($sParam1 == "precision"){
               if(is_numeric($sParam2) && $sParam2 >= 0)
                  $this->iPrecision = (int)$sParam2;
            }
            
            // Raster berechnen (colnum, colwidth, gutter)
            if($sParam1 == "grid"){
               if(preg_match("#([0-9.]+)\s+([0-9.]+)\s+([0-9.]+)(\s+[a-zA-Z_][a-zA-Z0-9_]*)*#si", $sParam2, $aMatches)){
                  // Anzahl der Spalten
                  $iColNum = $aMatches[1];
                  
                  // Breite der Spalte
                  $iColWidth = $aMatches[2];
                  
                  // Breite des Zwischenraums
                  $iGutterWidth = $aMatches[3];
                  
                  // Name des Grids
                  $sGridName = trim($aMatches[4]);
                  
                  // Defaultgridname auf 'grid' setzen
                  if(!$sGridName)
                     $sGridName = "grid";
                  
                  // Breite der Seite
                  $iPageWidth = $iColNum*($iColWidth+$iGutterWidth)-$iGutterWidth;
                  
                  // Seitenbreite in der Variable ablegen
                  $this->setVar($sGridName.'_pagewidth', $iPageWidth);

                  // Gutter in der Variable ablegen
                  $this->setVar($sGridName.'_gutter', $iGutterWidth);
                  
                  // Anzahl der Spalten in der Variable ablegen
                  $this->setVar($sGridName.'_colnum', $iColNum);
                  
                  // Anzahl der Spalten in der Variable ablegen
                  $this->setVar($sGridName.'_colwidth', $iColWidth);          
                  
                  // Spaltenbreiten berechnen
                  for($iCol = 1; $iCol < $iColNum + 1; $iCol++){
                     $iColSpan = $iCol*($iColWidth+$iGutterWidth) - $iGutterWidth;
                     
                     // Variable setzen
                     $this->setVar($sGridName.'_colspan'.$iCol, $iColSpan);
                  }
               }
            }
            
            // Layout berechnen (@set layout #Gridname# #colspan# #colspan# #colspan#;)
            if($sParam1 == "layout"){
               if(preg_match("#([a-zA-Z_][a-zA-Z0-9_]*)*\s+(\d+)\s+(\d+)\s+(\d+)#si", $sParam2, $aMatches)){
                  // Layoutgridname
                  $sLayoutGridName = $aMatches[1];
                  
                  // Anzahl der Gridspalten der linken Layout-Spalte
                  $iColNumLeft = $aMatches[2];
                  
                  // Anzahl der Gridspalten der mittleren Layout-Spalte
                  $iColNumMain = $aMatches[3];
                  
                  // Anzahl der Gridspalten der rechten Layout-Spalte
                  $iColNumRight = $aMatches[4];
                  
                  // Wenn die mittlere Layoutspalte die Anzahl von Gridspalten gleich Null hat, abbrechen 
                  if($iColNumMain < 1 || $iColNumLeft < 0 || $iColNumRight < 0)
                     return;
                  
                  // Layout Suffix
                  $sLayoutSuffix = '2';
                  
                  // Layout 12x wenn die linke Spalte definiert ist
                  if($iColNumLeft > 0)
                     $sLayoutSuffix = '1'.$sLayoutSuffix;
                     
                  // Layout x23 wenn die linke Spalte definiert ist
                  if($iColNumRight > 0)
                     $sLayoutSuffix .= '3';
                     
                  // Linke Spalte definieren
                  if(($iColNumLeft > 0) && $this->isVarSet($sLayoutGridName.'_colspan'.$iColNumLeft))
                     $this->setVar('layout'.$sLayoutSuffix.'_left', $this->getVar($sLayoutGridName.'_colspan'.$iColNumLeft));
                  
                  // Mittlere Spalte definieren
                  if($this->isVarSet($sLayoutGridName.'_colspan'.$iColNumMain))
                     $this->setVar('layout'.$sLayoutSuffix.'_main', $this->getVar($sLayoutGridName.'_colspan'.$iColNumMain));
                  
                  // Rechte Spalte definieren
                  if($iColNumRight > 0 && $this->isVarSet($sLayoutGridName.'_colspan'.$iColNumRight))
                     $this->setVar('layout'.$sLayoutSuffix.'_right', $this->getVar($sLayoutGridName.'_colspan'.$iColNumRight));
               }
            }
            
            // Einen Suffix automatisch anhängen (z.B. für Variable "grid_" oder "grid_*")
            if($sParam1 == "append"){
               // Nach welcher Variable soll was angehängt werden
               if(preg_match("#([a-zA-Z_][a-zA-Z0-9_]*)(\**)\s+(.+)#si", $sParam2, $aMatches)){
                  // Nach welcher Variable anfügen
                  $sAppedAfter = $aMatches[1];
                  
                  // Sternchen. Mit: Nur der Anfang des Strings wird verglichen. Ohne: Der komplette String
                  $sAsterisk = str_replace('**', '*', trim($aMatches[2]));
                  
                  // Was soll angefügt werden 
                  $sAppendWhat = $aMatches[3];
                  
                  // Variablen Teile parsen
                  $sAppendWhat = $this->extractVarComment($sAppendWhat, true);
                  
                  // Suffix im Array ablegen
                  $this->aSuffixForVar[$sAppedAfter] = array($sAppendWhat, ($sAsterisk == '*'));
               }
            }
            
            break;
      }
   }
   
   
   /**
    * Gibt alle Variablen aus
    * 
    * @author Leonid Lezner <leonid.lezner@shopdienst.com>
    */
   public function debug(){
      $sRet = '<h2>Dynamic CSS dump</h2>';
      $sRet .= '<ul class="dyncss-vars">';
      foreach($this->aVars as $var => $val)
         $sRet .= "<li>".$var.": ".$val."</li>";
      echo $sRet."</ul>";
   }
}